/*************************************************

gui.c
used by tw686x

**************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <getopt.h>		/* getopt_long() */

#include <fcntl.h>		/* low-level i/o */
#include <unistd.h>
#include <errno.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <asm/types.h>		/* for videodev2.h */
#include <linux/videodev2.h>

#include <gdk/gdk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <pthread.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/Xvlib.h>

#include "tw686x.h"
#include "videocap.h"
#include "gui.h"
#include "xv_init.h"

static gboolean
display_video (gpointer data)
{
    int i;
    channel_t * chl;
    gui_t *gui = (gui_t *) data;

    for( i=0; i<gui->video_count; i++ ) {
        if( gui->b_quit ) break;
        chl = &gui->channel[i];
        if( !chl->b_show ) continue;
        pthread_mutex_lock (&(chl->mutex));
        if( chl->b_update ) {
            XvPutImage (gui->dpy, gui->im_port, gui->win, gui->gc,
                    chl->xvimage, 0, 0, chl->width, chl->height,
                    chl->rect.left, chl->rect.top,
                    chl->rect.width, chl->rect.height);
            chl->b_update = FALSE;
        }
        pthread_mutex_unlock (&(chl->mutex));
    }

    return !gui->b_quit;
}

static gboolean
display_init (gui_t *gui)
{
    int i;
    channel_t * chl;

    for( i=0; i<gui->video_count; i++ ) {
        chl = &gui->channel[i];
        chl->xvimage =
            xv_create_ximage (gui, chl->width, chl->height, FMT_VIDEO_YUYV);
        if( chl==NULL ) {
            break;
        }
    }

    return (i==gui->video_count);
}

static void
display_set_position(gui_t* gui, int screen_width, int screen_height)
{
    channel_t *chl;
    int i=0, k = 0;
    int nWidth, nHeight;
    int nChannel;

    if( gui->show_single ) {
        nChannel = 1;
    }
    else {
        nChannel = (gui->show_begin<gui->show_end) ? (gui->show_end-gui->show_begin+1) :
                   (gui->show_end + gui->video_count-gui->show_begin);
    }

    if(nChannel==1)						{ k = 1; }
    else if(nChannel>1 && nChannel<=4)	{ k = 2; }
    else if(nChannel>4 && nChannel<=9)	{ k = 3; }
    else if(nChannel>9 && nChannel<=16)	{ k = 4; }
    else if(nChannel>16&& nChannel<=25)	{ k = 5; }
    else								{ k = 6; }

    dprintf( "display set to (%dx%d)\n", screen_width, screen_height );

    nWidth = screen_width / k;
    nHeight= screen_height / k;

    if( gui->show_single ) {
        chl = &gui->channel[gui->show_curr];
        chl->rect.left  = 1 + nWidth*(i%k);
        chl->rect.top   = 1 + nHeight*(i/k);
        chl->rect.width = nWidth;
        chl->rect.height= nHeight;
    }
    else if( gui->show_begin<=gui->show_end ) {
        for(i=gui->show_begin; i<=gui->show_end; i++) {
            chl = &gui->channel[i];
            chl->rect.left  = 1 + nWidth*(i%k);
            chl->rect.top   = 1 + nHeight*(i/k);
            chl->rect.width = nWidth;
            chl->rect.height= nHeight;
            dcprintf( chl, "display at (%d,%d,%d,%d)\r\n",
                      chl->rect.left, chl->rect.top,
                      chl->rect.left+nWidth, chl->rect.top+nHeight );
        }
    }
    else {
        for(i=gui->show_end; i<gui->video_count; i++) {
            chl = &gui->channel[i];
            chl->rect.left  = 1 + nWidth*(i%k);
            chl->rect.top   = 1 + nHeight*(i/k);
            chl->rect.width = nWidth;
            chl->rect.height= nHeight;
            dcprintf( chl, "display at (%d,%d,%d,%d)\r\n",
                      chl->rect.left, chl->rect.top,
                      chl->rect.left+nWidth, chl->rect.top+nHeight );
        }
        for(i=0; i<=gui->show_end; i++) {
            chl = &gui->channel[i];
            chl->rect.left  = 1 + nWidth*(i%k);
            chl->rect.top   = 1 + nHeight*(i/k);
            chl->rect.width = nWidth;
            chl->rect.height= nHeight;
            dcprintf( chl, "display at (%d,%d,%d,%d)\r\n",
                      chl->rect.left, chl->rect.top,
                      chl->rect.left+nWidth, chl->rect.top+nHeight );
        }
    }
}

static void
display_single(gui_t *gui, int screen_width, int screen_height)
{
    int i;

    gui->show_single = !gui->show_single;
    display_set_position( gui, screen_width, screen_height );

    if( gui->show_single ) {
        for( i=0; i<gui->video_count; i++ ) {
            gui->channel[i].b_show = (i==gui->show_curr);
        }
    }
    else if( gui->show_begin<=gui->show_end ) {
        for( i=gui->show_begin; i<=gui->show_end; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
    }
    else {
        for( i=gui->show_begin; i<gui->video_count; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
        for( i=0; i<=gui->show_end; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
    }
}

static void
display_fullscreen(gui_t *gui, GtkWidget * widget)
{
    int i;

    gui->show_fullscreen = !gui->show_fullscreen;

    if( gui->show_single ) {
        for( i=0; i<gui->video_count; i++ ) {
            gui->channel[i].b_show = (i==gui->show_curr);
        }
    }
    else if( gui->show_begin<=gui->show_end ) {
        for( i=gui->show_begin; i<=gui->show_end; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
    }
    else {
        for( i=gui->show_begin; i<gui->video_count; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
        for( i=0; i<=gui->show_end; i++ ) {
            gui->channel[i].b_show = TRUE;
        }
    }

    gtk_window_set_decorated(GTK_WINDOW(gui->wdt), !gui->show_fullscreen);

    if( gui->show_fullscreen ) {
        gtk_window_fullscreen( GTK_WINDOW(gui->wdt) );
    }
    else {
        gtk_window_unfullscreen( GTK_WINDOW(gui->wdt) );
    }
}

static int
display_get_active(gui_t* gui, int x, int y)
{
    int i, ret=-1;
    channel_t *chl;

    if( gui->show_single ) {
        ret = gui->show_curr;
    }
    else if( gui->show_begin<=gui->show_end ) {
        for( i=gui->show_begin; i<=gui->show_end; i++ ) {
            chl = &gui->channel[i];
            if( (x>=chl->rect.left) && (x<(chl->rect.left+chl->rect.width)) &&
                (y>=chl->rect.top) && (y<(chl->rect.top+chl->rect.height)) ) {
                ret = i;
                break;
            }
        }
    }
    else {
        for( i=gui->show_begin; i<gui->video_count; i++ ) {
            chl = &gui->channel[i];
            if( (x>=chl->rect.left) && (x<(chl->rect.left+chl->rect.width)) &&
                (y>=chl->rect.top) && (y<(chl->rect.top+chl->rect.height)) ) {
                ret = i;
                break;
            }
        }
        if( ret<0 ) {
            for( i=0; i<=gui->show_end; i++ ) {
                chl = &gui->channel[i];
                if( (x>=chl->rect.left) && (x<(chl->rect.left+chl->rect.width)) &&
                    (y>=chl->rect.top) && (y<(chl->rect.top+chl->rect.height)) ) {
                    ret = i;
                    break;
                }
            }
        }
    }

    return ret;
}

static gint
window_ondestroy (GtkWidget * widget, gpointer data)
{
    gui_t *gui = (gui_t *) data;
    gui->on_quit(gui);

    gtk_main_quit ();

    return FALSE;
}

static gint
configure_event (GtkWidget * widget, GdkEventConfigure * event, gpointer data)
{
    gui_t *gui = data;

    display_set_position( gui, widget->allocation.width, widget->allocation.height );

    return TRUE;
}

static gint
expose_event (GtkWidget * widget, GdkEventExpose * event, gpointer data)
{
    //dprintf( "expose event\n" );

    return FALSE;
}

static gboolean
on_button_press (GtkWidget* widget, GdkEventButton * event, gpointer data)
{
    gui_t *gui = data;
    int i;

    if( event->type == GDK_BUTTON_PRESS ) {
        if(event->button == 1) {
            i = display_get_active( gui, (int)event->x, (int)event->y );
            if( i>=0 ) {
                gui->show_curr = i;
            }
            dprintf( "select current video %d (%d,%d)\r\n", gui->show_curr,
                     (int)event->x, (int)event->y );
        }
    }
    else if( event->type == GDK_2BUTTON_PRESS ) {
        if( event->button==1 ) {
            dprintf( "double click event\r\n" );
            i = display_get_active( gui, (int)event->x, (int)event->y );
            if( i>=0 ) {
                gui->show_curr = i;
                display_single( gui, widget->allocation.width, widget->allocation.height );
            }
        }
        else if( event->button==3 ) {
            dprintf( "right double click event\r\n" );
            i = display_get_active( gui, (int)event->x, (int)event->y );
            if( i>=0 ) {
                gui->show_curr = i;
                display_fullscreen( gui, widget );
            }
        }
    }

    return TRUE;
}

static void
on_key_press (GtkWidget* widget, GdkEventKey * event, gpointer data)
{
    gui_t *gui = (gui_t *) data;

    switch(event->keyval)
    {
        case GDK_Escape:
            gui->on_quit(gui);
            gtk_main_quit();
            break;
        default:
            break;
    }
}

int
gui_start (gui_t * gui)
{
    GtkWidget *window;
    GtkWidget *drawing_area;
    GtkWidget *vbox;
    GdkCursor *cursor;

    gtk_init (NULL, NULL);
    gui->dpy = NULL;

    window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (window), "TW686X demo");
    gtk_window_set_default_size(GTK_WINDOW( window), SCREEN_WIDTH, SCREEN_HEIGHT);

    g_signal_connect (G_OBJECT (window), "destroy",
        G_CALLBACK (window_ondestroy), (gpointer) gui);
    g_signal_connect (G_OBJECT(window), "key-press-event",
        G_CALLBACK(on_key_press), (gpointer) gui);

    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (window), vbox);
    gtk_widget_show (vbox);

    drawing_area = gtk_drawing_area_new ();
    gtk_box_pack_start (GTK_BOX (vbox), drawing_area, TRUE, TRUE, 0);
    gtk_widget_show (drawing_area);

    gtk_widget_set_events (drawing_area, GDK_EXPOSURE_MASK
         | GDK_LEAVE_NOTIFY_MASK
         | GDK_BUTTON_PRESS_MASK
         | GDK_BUTTON_RELEASE_MASK
         | GDK_BUTTON1_MASK
         | GDK_BUTTON2_MASK
         | GDK_BUTTON3_MASK
         | GDK_KEY_PRESS_MASK
         | GDK_POINTER_MOTION_MASK
         | GDK_POINTER_MOTION_HINT_MASK);
    g_signal_connect (G_OBJECT (drawing_area), "expose_event",
        G_CALLBACK (expose_event), (gpointer) gui);
    g_signal_connect (G_OBJECT (drawing_area), "configure_event",
        G_CALLBACK (configure_event), (gpointer) gui);
    g_signal_connect (G_OBJECT (drawing_area), "button-press-event",
        G_CALLBACK (on_button_press), (gpointer) gui);

    gtk_widget_show (window);

    cursor = gdk_cursor_new (GDK_HAND2);
    gdk_window_set_cursor (window->window, cursor);

    gui->win = gdk_x11_drawable_get_xid (drawing_area->window);
    gui->dpy = gdk_x11_display_get_xdisplay (gtk_widget_get_display (drawing_area));
    gui->gc = XCreateGC (gui->dpy, gui->win, 0, NULL);
    xv_image_init (gui);

    if( !display_init(gui) ) {
        dprintf( "Init display failed\r\n" ) ;
        return -EINVAL;
    }

    gui->wdt = window;//drawing_area;
    XvStopVideo (gui->dpy, gui->im_port, gui->win);

    gui->on_start( gui );
    g_timeout_add ( 15, display_video, (gpointer) gui);

    gtk_main ();

    return 0;
}
